import invariant from 'tiny-invariant';
import { _TypedDataEncoder } from '@ethersproject/hash';
import { permit2Domain } from './domain';
import { MaxSigDeadline, MaxUnorderedNonce, MaxSignatureTransferAmount } from './constants';
const TOKEN_PERMISSIONS = [
    { name: 'token', type: 'address' },
    { name: 'amount', type: 'uint256' },
];
const PERMIT_TRANSFER_FROM_TYPES = {
    PermitTransferFrom: [
        { name: 'permitted', type: 'TokenPermissions' },
        { name: 'spender', type: 'address' },
        { name: 'nonce', type: 'uint256' },
        { name: 'deadline', type: 'uint256' },
    ],
    TokenPermissions: TOKEN_PERMISSIONS,
};
const PERMIT_BATCH_TRANSFER_FROM_TYPES = {
    PermitBatchTransferFrom: [
        { name: 'permitted', type: 'TokenPermissions[]' },
        { name: 'spender', type: 'address' },
        { name: 'nonce', type: 'uint256' },
        { name: 'deadline', type: 'uint256' },
    ],
    TokenPermissions: TOKEN_PERMISSIONS,
};
function permitTransferFromWithWitnessType(witness) {
    return Object.assign({ PermitWitnessTransferFrom: [
            { name: 'permitted', type: 'TokenPermissions' },
            { name: 'spender', type: 'address' },
            { name: 'nonce', type: 'uint256' },
            { name: 'deadline', type: 'uint256' },
            { name: 'witness', type: witness.witnessTypeName },
        ], TokenPermissions: TOKEN_PERMISSIONS }, witness.witnessType);
}
function permitBatchTransferFromWithWitnessType(witness) {
    return Object.assign({ PermitBatchWitnessTransferFrom: [
            { name: 'permitted', type: 'TokenPermissions[]' },
            { name: 'spender', type: 'address' },
            { name: 'nonce', type: 'uint256' },
            { name: 'deadline', type: 'uint256' },
            { name: 'witness', type: witness.witnessTypeName },
        ], TokenPermissions: TOKEN_PERMISSIONS }, witness.witnessType);
}
function isPermitTransferFrom(permit) {
    return !Array.isArray(permit.permitted);
}
export class SignatureTransfer {
    /**
     * Cannot be constructed.
     */
    constructor() { }
    // return the data to be sent in a eth_signTypedData RPC call
    // for signing the given permit data
    static getPermitData(permit, permit2Address, chainId, witness) {
        invariant(MaxSigDeadline.gte(permit.deadline), 'SIG_DEADLINE_OUT_OF_RANGE');
        invariant(MaxUnorderedNonce.gte(permit.nonce), 'NONCE_OUT_OF_RANGE');
        const domain = permit2Domain(permit2Address, chainId);
        if (isPermitTransferFrom(permit)) {
            validateTokenPermissions(permit.permitted);
            const types = witness ? permitTransferFromWithWitnessType(witness) : PERMIT_TRANSFER_FROM_TYPES;
            const values = witness ? Object.assign(permit, { witness: witness.witness }) : permit;
            return {
                domain,
                types,
                values,
            };
        }
        else {
            permit.permitted.forEach(validateTokenPermissions);
            const types = witness ? permitBatchTransferFromWithWitnessType(witness) : PERMIT_BATCH_TRANSFER_FROM_TYPES;
            const values = witness ? Object.assign(permit, { witness: witness.witness }) : permit;
            return {
                domain,
                types,
                values,
            };
        }
    }
    static hash(permit, permit2Address, chainId, witness) {
        const { domain, types, values } = SignatureTransfer.getPermitData(permit, permit2Address, chainId, witness);
        return _TypedDataEncoder.hash(domain, types, values);
    }
}
function validateTokenPermissions(permissions) {
    invariant(MaxSignatureTransferAmount.gte(permissions.amount), 'AMOUNT_OUT_OF_RANGE');
}
//# sourceMappingURL=signatureTransfer.js.map